/**********************************************************
**  fsFileStream.cpp
**
**      This is the windows version of the fsFileStream
**	implementation.
**
**	Gigawatt Studios
**	Copyright(C) 2000 - All Rights Reserved
\*********************************************************/
#include "stdafx.h"
#include "fsFileStream.hpp"

#include <windows.h>

#define INVALID_SET_FILE_POINTER ((DWORD)-1)

#include "fsFileUtil.hpp"

struct fsFileStreamImp
{
	HANDLE m_File;
	fsFileStream::AccessType m_Access;
	int m_FilePointer;
	std::string m_Locator;
};

namespace
{

//========================================================================
//========================================================================
void handle_windows_error(const std::string& i_Locator)
{
	// handle error in your favorite way, maybe throw exception
}

}

//========================================================================
//	Constructor.  This could throw an exception if the file doesn't
//	exist, or something.  So be ready!
//========================================================================
fsFileStream::fsFileStream(const std::string& i_Locator, AccessType i_DesiredAccess)
{
	DWORD access;
	DWORD share_mode;

	switch ( i_DesiredAccess )
	{
		case e_ReadOnly:
			access = GENERIC_READ;
			share_mode = FILE_SHARE_READ;
		break;

		case e_WriteOnly:
			access = GENERIC_WRITE;
			share_mode = 0;
		break;

		case e_ReadWrite:
			access = GENERIC_READ | GENERIC_WRITE;
			share_mode = 0;
		break;

		default:
		//	DBG_ASSERT0(false, "Invalid Access type");
		break;
	}

	HANDLE new_file = INVALID_HANDLE_VALUE;

	new_file = ::CreateFileA(	i_Locator.c_str(),
								access,
								share_mode,
								NULL,
								OPEN_EXISTING,
								FILE_ATTRIBUTE_NORMAL,
								NULL);

	if ( new_file == INVALID_HANDLE_VALUE )
	{
		handle_windows_error(i_Locator); // this will end up throwing, and abort construction
	}

	// save our work
	m_pImp = new fsFileStreamImp;
	m_pImp->m_File = new_file;
	m_pImp->m_Access = i_DesiredAccess;
	m_pImp->m_FilePointer = 0;
	m_pImp->m_Locator = i_Locator;
}

//========================================================================
//	Destructor
//========================================================================
fsFileStream::~fsFileStream()
{
	::CloseHandle(m_pImp->m_File);
	delete m_pImp;
}

//========================================================================
//	Read reads i_NumBytes into the buffer.  If the file is too short 
//	to read	i_NumBytes, it will read as many as it can.  The return
//	value is the number actually read into the o_Buffer.  This function
//	also advances the file pointer.
//========================================================================
int fsFileStream::Read(int i_NumBytes, void* o_Buffer)
{
	//DBG_ASSERT0(m_pImp->m_Access != e_WriteOnly, "Wrong access type for Read");

	DWORD num_read = 0;
	BOOL ret_val = 0;

	ret_val = ::ReadFile(	m_pImp->m_File,
							o_Buffer,
							DWORD(i_NumBytes),
							&num_read,
							NULL);

	if ( ret_val == 0 )
	{
		// handle failure
		handle_windows_error(m_pImp->m_Locator);
	}
	else
	{
		m_pImp->m_FilePointer += int(num_read);
	}

	return int(num_read);
}

//========================================================================
//	Write writes i_NumBytes from o_Buffer into the file.  The file 
//	pointer will also be advanced.
//========================================================================
void fsFileStream::Write(int i_NumBytes, const void* i_Buffer)
{
	//DBG_ASSERT0(m_pImp->m_Access != e_ReadOnly, "Wrong access type for Write");

	DWORD num_written = 0;
	BOOL ret_val = 0;

	ret_val = ::WriteFile(	m_pImp->m_File,
							i_Buffer,
							i_NumBytes,
							&(num_written),
							NULL);

	if ( ret_val == 0 )
	{
		// handle failure
		handle_windows_error(m_pImp->m_Locator);
	}
	else
	{
		m_pImp->m_FilePointer += int(num_written);
	}
}

//========================================================================
//	GetFilePos returns the current position of the file pointer.
//========================================================================
int fsFileStream::GetFilePos() const
{
	return m_pImp->m_FilePointer;
}

//========================================================================
//	GetLocator returns this filestream's associated locator
//========================================================================
const std::string& fsFileStream::GetLocator() const
{
	return m_pImp->m_Locator;
}


//========================================================================
//	SetFilePos sets the file pointer to the given value.  It is valid
//	to move the file pointer beyond the end of the file.
//========================================================================
void fsFileStream::SetFilePos(  int i_Pos, AccessPointType i_DesiredAccessPoint )
{
	DWORD ret_val = INVALID_SET_FILE_POINTER;
	DWORD move_method;

	if( i_DesiredAccessPoint == e_Beginning )
	{
		//DBG_ASSERT0( i_Pos >= 0,  "Negative seek attempted" );
		move_method = FILE_BEGIN;
	}
	else if( i_DesiredAccessPoint == e_Current )
	{
		//DBG_ASSERT0( m_pImp->m_FilePointer - i_Pos >= 0, "Negative seek attempted" );
		move_method = FILE_CURRENT;
	}
	else
	{
		move_method = FILE_END;
	}

	ret_val = ::SetFilePointer(	m_pImp->m_File,
								i_Pos,
								NULL,
								move_method );

	if ( ret_val == INVALID_SET_FILE_POINTER )
	{
		// handle failure
		handle_windows_error(m_pImp->m_Locator);
	}
	else
	{
		m_pImp->m_FilePointer = int(ret_val);
	}
	
}

//========================================================================
//	WriteHeader writes the given i_Header to the file.
//========================================================================
void fsFileStream::WriteHeader(const fsFileStream::Header& i_Header)
{
	//write signature
	this->Write(4, "GWFB");

	//set header's mode to this filebin's mode
	fsFileStream::Header Header = i_Header;
	Header.m_Mode = 2; // Little endian

	//write header
	this->Write(sizeof(fsFileStream::Header), &Header);
}
